% Model of a nonisothermal batch reactor - Alkylation
%                                          Two reactions

% Main file

% For more information, see:

% A. De Visscher 
% Lecture Notes in Chemical Engineering Kinetics and Chemical Reactor Design
% Createspace (2013) 

% (C) Alex De Visscher (2013)

% ---------------------------

% data

clear all

tspan = 0:5:400;     % We want output every 5 s up to 500 s
cC40 = 1;            % Initial butene concentration (kmol/m3)
ciC40 = 8.52;        % Initial isobutane concentration (kmol/m3)
ciC80 = 0;           % Initial isooctane concentration (kmol/m3)
ciC120 = 0;          % Initial isododecane concentration (kmol/m3)

% k1                 % Rate constant (m3/kg/s)
A = 265;
EA = 10000;

% K                  % Rate constant (-)
B = 30;
H = 15000;

V = 20;              % Reactor volume (m3)
m = 5;               % Catalyst mass (kg)
Ac = 100;            % Area of heating/cooling coil (m2)
U = 0.500;           % Heat transfer coefficient (kJ/m2 K)
Tc = 373.15;         % Temperature of heat transfer medium (K)
ws = 1000;           % shaft work (J)

DH0 = -65100;        % Enthalpy of reaction at 25 C
CpiC4 = 129;         % Heat capacity of isobutane (kJ/kmol K)
CpC4 = 124;          % Heat capacity of butene (kJ/kmol K)
CpiC8 = 247;         % Heat capacity of isooctane (kJ/kmol K)
DCp = CpiC8 - CpiC4 - CpC4;  % Heat capacity of reaction
CpiC12 = CpiC8 + CpC4 + DCp; % Heat capacity isododecane

c0(1) = cC40;        % Running variable for cC4 is C(:,1)
c0(2) = ciC40;       % Running variable for ciC4 is C(:,2)
c0(3) = ciC80;       % Running variable for ciC8 is C(:,3)
c0(4) = ciC120;      % Running variable for ciC12 is C(:,4)
c0(5) = 298.15;      % Running variable for T is C(:,5)

% Call of differential equation solver

options = odeset('RelTol', 1e-6, 'AbsTol', 1e-8, 'InitialStep', 0.01);
[T,C] = ode45(@f,tspan,c0,options,A,EA,B,H,V,m,Ac,U,Tc,ws,DH0,CpiC4,CpC4,CpiC8,CpiC12,DCp);

% Output

figure(1)
plot(T,C(:,1:4))
xlabel('Time (s)');
ylabel('Concentration (kmol/m3)');
figure(2)
plot(T,C(:,5))
xlabel('Time (s)');
ylabel('Temperature (K)');
T
C
[s1,s2] = size(T);
conversion = (C(1,1)-C(s1,1))/C(1,1)
conv_iC8 = C(s1,3)/C(1,1)

